/*
 * Controlling MCP3208 ADC by SPI (using spidev driver)
 *
 * Copyright (c) 2007  MontaVista Software, Inc.
 * Copyright (c) 2007  Anton Vorontsov <avorontsov@ru.mvista.com>
 *
 * Modifications 2012 by Claus Knel <info@ckuehnel.ch>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License.
 *
 * The orignal source is http://sourceforge.net/projects/raspberrypisnip/files/C/mcp3208.c/download
 *
 * 2014/7-2015/7
 * Modifications For getting by CSV type file and temperature measure of thermocouples by Hiroshima-gas TRI (AN).
 * Measure the getting Volage Amplifed minute voltage of thermocouples.
 *
 * (Example) Compile on Raspberry Pi by gcc tca8.c -o tca -lm
 */

#include <stdint.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/spi/spidev.h>
#include <time.h>
#include <math.h>

#include "alarm_mail8.h"

#define ARRAY_SIZE(a) (sizeof(a) / sizeof((a)[0]))

#define DEBUG 0

// Configuration bits for single-ended channel selection
#define CHANNEL0 0x060000   // Channel 0
#define CHANNEL1 0x064000   // Channel 1
#define CHANNEL2 0x068000   // Channel 2
#define CHANNEL3 0x06C000   // Channel 3
#define CHANNEL4 0x070000   // Channel 4
#define CHANNEL5 0x074000   // Channel 5
#define CHANNEL6 0x078000   // Channel 6
#define CHANNEL7 0x07C000   // Channel 7

static const char *device0 = "/dev/spidev0.0";
static const char *device1 = "/dev/spidev0.1";
static uint8_t mode;
static uint8_t bits = 8;
static uint32_t speed = 500000;
static uint16_t delay;

uint32_t channel;
char currentdir[100] = "/usr/local/measure/tc";
static const int OPT_R_MAX = 1800;			//measure interval max 1800 sec (30 minutes)
static const int cenr = 1;				//using 0:CE0 1:CE1
char mbody[10][255];					//alarm mail body part
static const int mag_opamp = 200;			//LTK001 operational amplifier magnification

FILE *fp10;
char *fname10 = "../set/tca.setting";		//analog measure setting file
int tcachno[8];    				//Channel number(0-7) and Instrumentation signal type 0:0-5V 1:1-5V
int tcatype[8];
char tcaname[8][31]; 				//the measure name of every channel
int tcaspanrev[8];				//span-control of every channnel:Standard 5000
int tcaat[8];					//Analog alarm type 0:none 1:upper limit 2:lower limit 3:upper and lower limits
float tcaul[8];					//Analog upper limit
float tcall[8];					//Analog lower limit
float tcavalue[8];				//the calculated temperature value (unit *c)
double tcav;

FILE *fp20;
char *fname20 = "../set/measure.setting";	//general measure setting file
int intervalsec[4];				//analog measure interval(sec)
char tca_name[4][51];				//analog measure project name


FILE *fp30;
char *fname30 = "tca_header.txt";		//header text of logging data

FILE *fp40;
char *fname40 = "../tempdata/tca.csv";		//logging data before FTP upload
char tcatime[100];				//the time of measure

FILE *fp50;
char *fname50 = "tcastart.mem";			//memory about amd setting

FILE *fp70;
char *fname70 = "../log/alarm_log/tca_alarm.log";	//alarm logging file
time_t date_old[8][3];					//time-memories when alarm occurrence
int moa[8][3];						//memories of alarm every channel
int uorl;						//memory of alarm detecrion
time_t nowtime;						//time of alarm occurrence
time_t mailtime[8];					//memory of alarm-mail sending time
int sec_time;						//prohibition period of a next send mail
int firsta[8];						//memory of the first sending alarm mail

/* recording alarm */
int alarm_record(int chno,int uorl) {

    int rows = 0,rowsx =0 ;
    char buffx[256];

    fp70 = fopen (fname70,"a");
    if (fp70 == NULL) {
	printf("%s ファイルが開けません\n",fname70);
	return -1;
    }

    switch (uorl) {
	case 1:
		fprintf (fp70,"TCA_Ch%d,%s,Upper level alarm,%s,%d,%g,%g,",tcachno[chno],tcatime,&tcaname[chno],tcaat[chno],tcaul[chno],tcall[chno]);
		break;
	case 2:
		fprintf (fp70,"TCA_Ch%d,%s,Lower level alarm,%s,%d,%g,%g,",tcachno[chno],tcatime,&tcaname[chno],tcaat[chno],tcaul[chno],tcall[chno]);
		break;
    }
    if (tcavalue[chno] >= 100) {
	fprintf (fp70,"%.0f,*C\n",tcavalue[chno]);
    } else {
	fprintf (fp70,"%0.1f,*C\n",tcavalue[chno]);
    }

   if ((moa[chno][0] == 1) && (moa[chno][1] == 1) && (moa[chno][2] == 1)) {
	date_old[chno][0] = date_old[chno][1];
	date_old[chno][1] = date_old[chno][2];
	date_old[chno][2] = nowtime;
    }

    if ((moa[chno][0] == 1) && (moa[chno][1] == 1) && (moa[chno][2] == 0)) {
	moa[chno][2] = 1;
	date_old[chno][2] = nowtime;
    }

    if ((moa[chno][0] == 1) && (moa[chno][1] == 0)) {
	moa[chno][1] = 1;
	date_old[chno][1] = nowtime;
    }

    if (moa[chno][0] == 0) {
	moa[chno][0] = 1;
	date_old[chno][0] = nowtime;
    }

    if (intervalsec[2] <= 180) sec_time = 1800;
    if ((intervalsec[2] > 180) && (intervalsec[2] <= 600)) sec_time = 3600;
    if (intervalsec[2] >= 600) sec_time = 7200;

    if ((moa[chno][2] == 1) && (date_old[chno][2]-date_old[chno][0] <= sec_time)) {

	sprintf(mbody[0], "温度計測A「 %s 」にて %d (秒) 以内に3回以上の警報が発生しています。\n", &tca_name[2],sec_time);
	sprintf(mbody[1], "\n");
	sprintf(mbody[2], "----------------直近のアラームデータ----------------\n");
	switch(uorl) {
		case 1:
			sprintf(mbody[3],"「上限警報発生」  日時: %s\n",tcatime);break;
		case 2:
 			sprintf(mbody[3],"「下限警報発生」  日時: %s\n",tcatime);break;
	}
	sprintf(mbody[4],"TCA_Ch%d : %s\n",tcachno[chno],&tcaname[chno]);
	sprintf(mbody[5],"警報発生時の測定値 :");
	if (tcavalue[chno] >= 100) {
		sprintf (mbody[5],"%.0f(*C)\n",tcavalue[chno]);
	} else {
		sprintf (mbody[5],"%0.1f(*C)\n",tcavalue[chno]);
	}
	sprintf(mbody[6],"\n");
	sprintf(mbody[7],"----------------------警報設定値---------------------\n");
	switch (tcaat[chno]) {
		case 1:
			sprintf(mbody[8],"上限警報: %g(*C)\n",tcaul[chno]);break;
		case 2:
			sprintf(mbody[8],"下限警報: %g(*C)\n",tcall[chno]);break;
		case 3:
			sprintf(mbody[8],"上下限警報:上限 %g(*C),下限 %g(*C)\n",tcaul[chno],tcall[chno]);
			break;
	}

	if (firsta[chno] == 0) {
		alarm_mail(mbody);
		firsta[chno] = 1;
		mailtime[chno] = nowtime;
		fprintf (fp70,"%s,sent alarm mail\n",tcatime);
		moa[chno][0]=0;
		moa[chno][1]=0;
		moa[chno][2]=0;
	}

	if (firsta[chno] != 0) {
		if (mailtime[chno] + sec_time <= nowtime) {
			alarm_mail(mbody);
			mailtime[chno] = nowtime;
			fprintf (fp70,"%s,sent alarm mail\n",tcatime);
			moa[chno][0]=0;
			moa[chno][1]=0;
			moa[chno][2]=0;
		}
	}
    }
	/* Adjust alarm-log length */

	while (fgets( buffx,256, fp70) != NULL ) rows++;		//count rows of alarm-log
	fclose(fp70);
	rowsx = rows;

	if (rowsx >= 500) {
		FILE *fp75;
		char *fname75="../log/alarm_log/adjusttca.txt";	//temporary file for alarm log adjusting
		fp70 = fopen(fname70,"r");
		fp75 = fopen(fname75,"w");
		rows =0;

		while (fgets (buffx,256,fp70) != NULL ){
			if (rows >= rowsx -200) fprintf (fp75,buffx);
				rows++;
		}

		fclose(fp70);
		fclose(fp75);
		remove (fname70);
		rename (fname75,fname70);
		remove (fname75);
	}

}


static void pabort(const char *s)
{
	perror(s);
	abort();
}

static void transfer(int fd,int tcachno2,int tcatype2,int spanrev)
{
	uint16_t adc;
	uint8_t ret,i;

	uint8_t tx[3]; 		// 3 bytes will be sent to MPC3208
	uint8_t rx[ARRAY_SIZE(tx)] = {0, };

	float adcmv;
	double tcavx;

	for (i = 0; i < ARRAY_SIZE(tx); i++)
  	{
      	tx[ARRAY_SIZE(tx)-i-1] = channel%256;
      	channel = channel >> 8;
  	}

	struct spi_ioc_transfer tr =
	{
		.tx_buf = (unsigned long)tx,
		.rx_buf = (unsigned long)rx,
		.len = ARRAY_SIZE(tx),
		.delay_usecs = delay,
		.speed_hz = speed,
		.bits_per_word = bits,
	};

	ret = ioctl(fd, SPI_IOC_MESSAGE(1), &tr);
	if (ret < 1)
		pabort("can't send spi message");

	adc = (rx[1] << 8) + rx[2];
	adc &= 0xFFF;


	adcmv = adc *spanrev/4096;
        //adcmv = adc * 5000/4096;

	tcavx = adcmv/mag_opamp;

	switch (tcatype2) {
	case 1:
		if ((tcavx >= 0) && (tcavx < 16.397)) {
			tcav = 0.032676+25.44687*tcavx-0.575287071*pow(tcavx,2)+0.10692713*pow(tcavx,3)-0.00758*pow(tcavx,4);
		} else if (tcavx >= 16.397) {
			tcav = -51.7772+34.59742*tcavx-0.691302287*pow(tcavx,2)+0.01996650*pow(tcavx,3)-0.00027*pow(tcavx,4);
		} else {
			tcav = 9999;
		}
		break;
	case 2:
		if ((tcavx >= 0) && (tcavx < 20.872)) {
			tcav = 0.029492809+25.83012512*tcavx-0.706419007*pow(tcavx,2)+0.03532747*pow(tcavx,3)-0.001131265*pow(tcavx,4)+0.0000156779*pow(tcavx,5);
		} else {
			tcav = 9999;
		}
		break;
	}

}


int transfer2()
{
int ret = 0;
int fd;
int j,j2;
double tcavaluex = 0 ;

	switch (cenr) {
		case 0:fd = open(device0, O_RDWR);break;
		case 1:fd = open(device1, O_RDWR);break;
	}

	if (fd < 0)
		pabort("can't open device");

	 // max speed hz
	ret = ioctl(fd, SPI_IOC_WR_MAX_SPEED_HZ, &speed);
	if (ret == -1)
		pabort("can't set max speed hz");

	ret = ioctl(fd, SPI_IOC_RD_MAX_SPEED_HZ, &speed);
	if (ret == -1)
		pabort("can't get max speed hz");

#if DEBUG
	printf("spi device: %s\n", device);

	printf("spi mode: %d\n", mode);
	printf("bits per word: %d\n", bits);
	printf("max speed: %d Hz (%d KHz)\n", speed, speed/1000);
#endif

	fp40 = fopen(fname40, "a" );
  	if( fp40 == NULL ){
        	printf( "%s ファイルが開けません\n", fname40 );
        	return -1;
  	}

	for (j=0; j<8; j++){
		tcavaluex = 0 ;
		for (j2=0;j2<4;j2++){

			switch (j){
				case 0:channel = CHANNEL0;break;
				case 1:channel = CHANNEL1;break;
				case 2:channel = CHANNEL2;break;
				case 3:channel = CHANNEL3;break;
				case 4:channel = CHANNEL4;break;
				case 5:channel = CHANNEL5;break;
				case 6:channel = CHANNEL6;break;
				case 7:channel = CHANNEL7;break;
			}
			transfer(fd,tcachno[j],tcatype[j],tcaspanrev[j]);
			tcavaluex = tcavaluex + tcav;
			usleep(40);
	   	}
		tcavalue[j]=tcavaluex/4;
	}


	for (j=0; j<8; j++) {
		if (j == 0) fprintf(fp40,"%s,",tcatime);
		if (tcavalue[j] >= 100) fprintf(fp40,"%.0f",tcavalue[j]);
		if (tcavalue[j] < 100) fprintf(fp40,"%0.1f",tcavalue[j]);
		if (j < 7 ) {
			fprintf(fp40,",");
		} else {
			fprintf(fp40,"\n");
		}

	// Alarm detection

		switch (tcaat[j]) {
			case 1:
				if (tcavalue[j] > tcaul[j]) alarm_record(j,1);
				break;
 			case 2:
				if (tcavalue[j] < tcall[j]) alarm_record(j,2);
				break;
			case 3:
				if (tcavalue[j] > tcaul[j]) alarm_record(j,1);
				if (tcavalue[j] < tcall[j]) alarm_record(j,2);
				break;
		}

	}

	fclose(fp40);
	close(fd);

}

/* read setting data */
int readsetting()
{
 int ret10,i10;
 int ret20,i20;
 int xxx1[4],xxx2[4];

  fp10 = fopen( fname10, "r" );
  if( fp10 == NULL ){
	printf( "%s ファイルが開けません\n", fname10 );
	return -1;
  }
  for (i10 = 0; i10 < 8; i10++)
        {
        ret10 = fscanf( fp10, "%d,%d,%[^,],%d,%d,%f,%f", &tcachno[i10], &tcatype[i10], &tcaname[i10],&tcaspanrev[i10],&tcaat[i10],&tcaul[i10],&tcall[i10]);
        }
  fclose (fp10);

  fp20 = fopen( fname20, "r" );
  if( fp20 == NULL ){
	printf( "%s ファイルが開けません\n", fname20 );
	return -1;
  }

  for (i20=0;i20<4;i20++) {
	ret20 = fscanf( fp20, "%d,%d,%[^,],%d", &xxx1[i20], &intervalsec[i20], &tca_name[i20][0] ,&xxx2[i20] );
  }
  fclose(fp20);
  if (intervalsec[2] > OPT_R_MAX) {
  	printf("計測間隔設定エラー\n");
  return -1;
  }

}



/* make header of csv data file */
int makeheader(){

  char *str30 = "ch_NO,tc_type,tca_measure_point,tcaspanrev,alarm_type,upper_limit,lower_limit\n";
  char *str31 = "* Thermocouple type 1:K 2:T\n";
  char *str32 = "* Alarm_type 0:None 1:Upper limit alarm 2:Lower limit alarm 3:Upper and lower limits\n";
  char *str33 = "* Span revison level of every channel (usually 5000)\n";
  int i10;

  fp30 = fopen(fname30,"w");
  if (fp30 == NULL ){
	printf("%s ファイルが開けません\n",fname30);
	return -1;
  }

  fprintf(fp30, "%s",&tca_name[0]);
  fputs("\n",fp30);
  fputs("\n",fp30);

  fputs (str30,fp30);
  for (i10 = 0; i10 < 8; i10++)
	{
  	fprintf(fp30, "%d,%d,%s,%d,%d,%g,%g", tcachno[i10], tcatype[i10], &tcaname[i10], tcaspanrev[i10],tcaat[i10],tcaul[i10],tcall[i10]);
  	fputs("\n",fp30);
	}

  fputs("\n",fp30);
  fputs (str31,fp30);
  fputs (str32,fp30);
  fputs (str33,fp30);
  fputs("\n",fp30);

  fprintf(fp30, " ,Ch%d,Ch%d,Ch%d,Ch%d,Ch%d,Ch%d,Ch%d,Ch%d\n",tcachno[0],tcachno[1],tcachno[2],tcachno[3],tcachno[4],tcachno[5],tcachno[6],tcachno[7]);
  fprintf(fp30, "%s,%s,%s,%s,%s,%s,%s,%s,%s\n", "Date", &tcaname[0],&tcaname[1],&tcaname[2],&tcaname[3],&tcaname[4],&tcaname[5],&tcaname[6],&tcaname[7] );
  fprintf(fp30, " ,(*C),(*C),(*C),(*C),(*C),(*C),(*C),(*C)\n");

  fclose (fp30);

  return 0;
}


int main(int argc, char *argv[])
{
   int ssignal;

   time_t timer;
   struct tm *date;
   char str2[256];

   time_t last = time(0);
   time_t next;

   int shour,smin,ssec;
   int pastsec1 = 0 ;
   int pastsec2 = 0 ;

   chdir(currentdir);
   readsetting();
   makeheader();

// order to be started measure loop
   while(1) {
	timer=time(NULL);
	date = localtime(&timer);
	strftime (str2,255,"%H",date);
	shour=atoi(str2);
       	strftime (str2,255,"%M",date);
        smin=atoi(str2);
       	strftime (str2,255,"%S",date);
        ssec=atoi(str2);
	pastsec1 = shour*3600 + smin * 60 + ssec;
	if (pastsec1%intervalsec[2] == 0) break;
   }

   while(1){
  	if (time(&next) != last){
     		last = next;
     		pastsec2++;
		if (pastsec2 == 1500) {		//remake header every interval 25 minutes (1500 sec)
			readsetting();
			makeheader();
			pastsec2 = 0;
		}

	        timer = time(NULL);
        	date = localtime(&timer);
        	strftime (str2,255,"%H",date);
        	shour=atoi(str2);
        	strftime (str2,255,"%M",date);
        	smin=atoi(str2);
        	strftime (str2,255,"%S",date);
        	ssec=atoi(str2);
        	pastsec1 = shour*3600 + smin * 60 + ssec;

		if (pastsec1%intervalsec[2] == 0 ) {
			strftime(tcatime, 100, "%Y/%m/%d %H:%M:%S", date);
			nowtime = timer;
			transfer2();
		}
		usleep(1000000-5);
	}

	fp50 = fopen(fname50,"r");
	if (fp50 == NULL) {
		printf("%s ファイルを開けません\n",fname50);
	}
	fscanf (fp50,"%d",&ssignal);
	fclose (fp50);
	if (ssignal == 0 ) break;

   }

chdir("/");
return 0;
}
